/**
* Nanotec Nanolib example
* Copyright (C) Nanotec GmbH & Co. KG - All Rights Reserved
*
* This product includes software developed by the
* Nanotec GmbH & Co. KG (http://www.nanotec.com/).
*
* The Nanolib interface headers and the examples source code provided are 
* licensed under the Creative Commons Attribution 4.0 Internaltional License. 
* To view a copy of this license, 
* visit https://creativecommons.org/licenses/by/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* The parts of the library provided in binary format are licensed under 
* the Creative Commons Attribution-NoDerivatives 4.0 International License. 
* To view a copy of this license, 
* visit http://creativecommons.org/licenses/by-nd/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*
* @file   MenuUtils.java
*
* @brief  Definition of CLI menu specific classes
*
* @date   29-10-2024
*
* @author Michael Milbradt
*
*/
package com.nanotec.example.NanolibExample;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;

import com.nanotec.nanolib.OdIndex;
import com.nanotec.nanolib.ResultDeviceId;
import com.nanotec.nanolib.ResultObjectDictionary;
import com.nanotec.nanolib.helper.LibraryLoader;
import com.nanotec.example.NanolibExample.MenuColor.ColorModifier;
import com.nanotec.nanolib.BusHardwareId;
import com.nanotec.nanolib.LogLevel;
import com.nanotec.nanolib.LogModule;
import com.nanotec.nanolib.LogModuleConverter;
import com.nanotec.nanolib.DeviceId;
import com.nanotec.nanolib.DeviceHandle;
import com.nanotec.nanolib.NanoLibAccessor;
import com.nanotec.nanolib.Nanolib;
import com.nanotec.nanolib.ObjectDictionary;
import com.nanotec.nanolib.BusHardwareOptions;
import com.nanotec.nanolib.BusHwOptionsDefault;
import com.nanotec.nanolib.LogLevelConverter;

/**
 * @brief Container class for menu
 */
public class MenuUtils {

    // use only one InputStreamReader for example
    private static InputStreamReader inputStreamReader = new InputStreamReader(System.in);
    // reader for user input
    public static BufferedReader reader = new BufferedReader(inputStreamReader);

    /**
     * @brief Inner class for constants
     */
    public static class NlcConstants {

        // OD Indexes
        public static final OdIndex OD_SI_UNIT_POSITION = new OdIndex(0x60A8, (short)0x00);
        public static final OdIndex OD_CONTROL_WORD = new OdIndex(0x6040, (short)0x00);
        public static final OdIndex OD_STATUS_WORD = new OdIndex(0x6041, (short)0x00);
        public static final OdIndex OD_HOME_PAGE = new OdIndex(0x6505, (short)0x00);
        public static final OdIndex OD_NANOJ_CONTROL = new OdIndex(0x2300, (short)0x00);
        public static final OdIndex OD_NANOJ_STATUS = new OdIndex(0x2301, (short)0x00);
        public static final OdIndex OD_NANOJ_ERROR = new OdIndex(0x2302, (short)0x00);
        public static final OdIndex OD_MODE_OF_OPERATION = new OdIndex(0x6060, (short)0x00);
        public static final OdIndex OD_TARGET_VELOCITY = new OdIndex(0x60FF, (short)0x00);
        public static final OdIndex OD_PROFILE_VELOCITY = new OdIndex(0x6081, (short)0x00);
        public static final OdIndex OD_TARGET_POSITION = new OdIndex(0x607A, (short)0x00);
        public static final int OD_ERROR_STACK_INDEX = 0x1003;
        public static final OdIndex OD_ERROR_COUNT = new OdIndex(0x1003, (short)0x00);
        public static final OdIndex OD_POS_ENCODER_INCREMENTS_INTERFACE_1 = new OdIndex(0x60E6, (short)0x1);
        public static final OdIndex OD_POS_ENCODER_INCREMENTS_INTERFACE_2 = new OdIndex(0x60E6, (short)0x2);
        public static final OdIndex OD_POS_ENCODER_INCREMENTS_INTERFACE_3 = new OdIndex(0x60E6, (short)0x3);
        public static final OdIndex OD_MOTOR_DRIVE_SUBMODE_SELECT = new OdIndex(0x3202, (short)0x00);
        public static final OdIndex OD_STORE_ALL_PARAMS = new OdIndex(0x1010, (short)0x01);
        public static final OdIndex OD_RESTORE_ALL_DEF_PARAMS = new OdIndex(0x1011, (short)0x01);
        public static final OdIndex OD_RESTORE_TUNING_DEF_PARAMS = new OdIndex(0x1011, (short)0x06);
        public static final OdIndex OD_MODE_OF_OPERATION_DISPLAY = new OdIndex(0x6061, (short)0x00);

        // Bus Hardware Menu Texts
        public static final String BUS_HARDWARE_MENU = "Bus Hardware Menu";
        public static final String BUS_HARDWARE_OPEN_MI = "Open Bus Hardware";
        public static final String BUS_HARDWARE_CLOSE_MI = "Close bus hardware";
        public static final String BUS_HARDWARE_SCAN_MI = "Scan for Bus hardware";
        public static final String BUS_HARDWARE_CLOSE_ALL_MI = "Close all bus hardware";

        // Device Menu Texts
        public static final String DEVICE_MENU = "Device Menu";
        public static final String DEVICE_SCAN_MI = "Scan for Devices";
        public static final String DEVICE_CONNECT_MENU = "Connect to device Menu";
        public static final String DEVICE_DISCONNECT_MENU = "Disconnect from device Menu";
        public static final String DEVICE_SELECT_ACTIVE_MENU = "Select active device";
        public static final String DEVICE_REBOOT_MI = "Reboot device";
        public static final String DEVICE_UPDATE_FW_MI = "Update firmware";
        public static final String DEVICE_UPDATE_BL_MI = "Update bootloader";
        public static final String DEVICE_UPLOAD_NANOJ_MI = "Upload NanoJ program";
        public static final String DEVICE_RUN_NANOJ_MI = "Run NanoJ program";
        public static final String DEVICE_STOP_NANOJ_MI = "Stop NanoJ program";

        // Device Information Menu Texts
        public static final String DEVICE_INFORMATION_MENU = "Device information Menu";
        public static final String DEVICE_GET_VENDOR_ID_MI = "Read vendor Id";
        public static final String DEVICE_GET_PRODUCT_CODE_MI = "Read product code";
        public static final String DEVICE_GET_DEVICE_NAME_MI = "Read device name";
        public static final String DEVICE_GET_HW_VERSION_MI = "Read device hardware version";
        public static final String DEVICE_GET_FW_BUILD_ID_MI = "Read device firmware build id";
        public static final String DEVICE_GET_BL_BUILD_ID_MI = "Read device bootloader build id";
        public static final String DEVICE_GET_SERIAL_NUMBER_MI = "Read device serial number";
        public static final String DEVICE_GET_UNIQUE_ID_MI = "Read device unique id";
        public static final String DEVICE_GET_BL_VERSION_MI = "Read device bootloader version";
        public static final String DEVICE_GET_HW_GROUP_MI = "Read device hardware group";
        public static final String DEVICE_GET_CON_STATE_MI = "Read device connection state";
        public static final String DEVICE_GET_ERROR_FIELD_MI = "Read device error field";
        public static final String DEVICE_RESTORE_ALL_DEFAULT_PARAMS_MI = "Restore all default parameters";

        // OD Interface Menu Texts
        public static final String OD_INTERFACE_MENU = "Object Dictionary Interface Menu";
        public static final String OD_ASSIGN_OD_MI = "Assign an object dictionary to active device (e.g. od.xml)";
        public static final String OD_READ_NUMBER_MI = "readNumber (raw, untyped)";
        public static final String OD_READ_STRING_MI = "readString";
        public static final String OD_READ_BYTES_MI = "readBytes (raw, untyped)";
        public static final String OD_WRITE_NUMBER_MI = "writeNumber (data bitlength needed)";
        public static final String OD_READ_NUMBER_VIA_OD_MI = "readNumber (via OD interface, get type information)";
        public static final String OD_WRITE_NUMBER_VIA_OD_MI = "writeNumber (via OD interface, no data bitlength needed)";

        // Logging Menu Texts
        public static final String LOGGING_MENU = "Logging Menu";
        public static final String LOGGING_SET_LOG_LEVEL_MI = "Set log level";
        public static final String LOGGING_SET_LOG_CALLBACK_MI = "Set logging callback";

        // Log Level Menu Texts
        public static final String LOG_LEVEL_MENU = "Log level Menu";
        public static final String LOG_LEVEL_TRACE_MI = "Set log level to 'Trace'";
        public static final String LOG_LEVEL_DEBUG_MI = "Set log level to 'Debug'";
        public static final String LOG_LEVEL_INFO_MI = "Set log level to 'Info'";
        public static final String LOG_LEVEL_WARN_MI = "Set log level to 'Warning'";
        public static final String LOG_LEVEL_ERROR_MI = "Set log level to 'Error'";
        public static final String LOG_LEVEL_CRITICAL_MI = "Set log level to 'Critical'";
        public static final String LOG_LEVEL_OFF_MI = "Set log level to 'Off'";

        // Logging Callback Menu Texts
        public static final String LOG_CALLBACK_MENU = "Logging Callback Menu";
        public static final String LOG_CALLBACK_CORE_MI = "Activate log callback for Nanolib Core";
        public static final String LOG_CALLBACK_CANOPEN_MI = "Activate log callback for CANopen module";
        public static final String LOG_CALLBACK_ETHERCAT_MI = "Activate log callback for EtherCAT module";
        public static final String LOG_CALLBACK_MODBUS_MI = "Activate log callback for Modbus module";
        public static final String LOG_CALLBACK_REST_MI = "Activate log callback for REST module";
        public static final String LOG_CALLBACK_USB_MI = "Activate log callback for USB/MSC module";
        public static final String LOG_CALLBACK_DEACTIVATE_MI = "Deactivate current log callback";

        // Sampler Menu Texts
        public static final String SAMPLER_EXAMPLE_MENU = "Sampler Example Menu";
        public static final String SAMPLER_NORMAL_WO_NOTIFY_MI = "Sampler w/o Notification - Normal Mode";
        public static final String SAMPLER_REPETETIVE_WO_NOTIFY_MI = "Sampler w/o Notification - Repetetive Mode";
        public static final String SAMPLER_CONTINUOUS_WO_NOTIFY_MI = "Sampler w/o Notification - Continuous Mode";
        public static final String SAMPLER_NORMAL_WITH_NOTIFY_MI = "Sampler with Notification - Normal Mode";
        public static final String SAMPLER_REPETETIVE_WITH_NOTIFY_MI = "Sampler with Notification - Repetetive Mode";
        public static final String SAMPLER_CONTINUOUS_WITH_NOTIFY_MI = "Sampler with Notification - Continuous Mode";

        // Motor Example Menu Texts
        public static final String MOTOR_EXAMPLE_MENU = "Motor Example Menu";
        public static final String MOTOR_AUTO_SETUP_MI = "Initial commissioning - motor auto setup";
        public static final String MOTOR_VELOCITY_MI = "Run a motor in profile velocity mode";
        public static final String MOTOR_POSITIONING_MI = "Run a motor in positioning mode";

        // Profinet Menu Texts
        public static final String PROFINET_EXAMPLE_MI = "ProfinetDCP example";

        // Main Menu Title
        public static final String MAIN_MENU = "Nanolib Example Main";
    }

    /**
     * @brief Inner class for menu context
     */
    public static class Context {
        public int selectedOption; // the selected option of user
        public String errorText; // the error text of last action (if error occurred)
        public LogLevel currentLogLevel; // holds the current log level
        public NanoLibAccessor nanolibAccessor; // holds the nanolib accessor
        public List<BusHardwareId> scannedBusHardwareIds; // holds found bus hardware ids
        public List<BusHardwareId> openableBusHardwareIds; // holds found bus hardware ids not yet opened
        public List<BusHardwareId> openBusHardwareIds; // holds opened bus hardware ids
        public List<DeviceId> scannedDeviceIds; // holds found devices of opened bus hardware
        public List<DeviceId> connectableDeviceIds; // holds found devices not yet connected
        public List<DeviceHandle> connectedDeviceHandles; // holds device handles of connected devices
        public DeviceHandle activeDevice; // the current active device
        public LogModule currentLogModule; // the log module currently used
        public boolean loggingCallbackActive; // flag for active logging callback
        public boolean waitForUserConfirmation; // flag to wait for user confirmation after a function has been executed
        public LoggingCallbackExample loggingCallback; // pointer to logging callback
        public ScanBusCallbackExample scanBusCallback; // pointer to scan bus callback
        public DataTransferCallbackExample dataTransferCallback; // pointer to data transfer callback
        public ColorModifier red = new ColorModifier(MenuColor.Code.FG_RED); // color modifier for red foreground color
        public ColorModifier green = new ColorModifier(MenuColor.Code.FG_GREEN); // color modifier for green foreground color
        public ColorModifier blue = new ColorModifier(MenuColor.Code.FG_BLUE); // color modifier for blue foreground color
        public ColorModifier yellow = new ColorModifier(MenuColor.Code.FG_YELLOW); // color modifier for yellow foreground color
        public ColorModifier lightRed = new ColorModifier(MenuColor.Code.FG_LIGHT_RED); // color modifier for light red foreground color
        public ColorModifier lightGreen = new ColorModifier(MenuColor.Code.FG_LIGHT_GREEN); // color modifier for light green foreground color 
        public ColorModifier lightBlue = new ColorModifier(MenuColor.Code.FG_LIGHT_BLUE); // color modifier for light blue foreground color
        public ColorModifier lightYellow = new ColorModifier(MenuColor.Code.FG_LIGHT_YELLOW); // color modifier for light yellow foreground color
        public ColorModifier darkGray = new ColorModifier(MenuColor.Code.FG_DARK_GRAY); // color modifier for dark gray foreground color
        public ColorModifier def = new ColorModifier(MenuColor.Code.FG_DEFAULT); // color modifier for the default color
        public ColorModifier resetAll = new ColorModifier(MenuColor.Code.RESET); // color modifier to reset everything to default

        /**
         * @brief Default constructor
         */
        public Context() {
            try {
                this.nanolibAccessor = LibraryLoader.setup();
            } catch (IOException e) {
                System.out.println(e.getMessage());
                System.exit(-1);
            }

            this.scannedBusHardwareIds = new ArrayList<>();
            this.openableBusHardwareIds = new ArrayList<>();
            this.openBusHardwareIds = new ArrayList<>();
            this.scannedDeviceIds = new ArrayList<>();
            this.connectableDeviceIds = new ArrayList<>();
            this.connectedDeviceHandles = new ArrayList<>();
            this.loggingCallback = null;
            this.scanBusCallback = null;
            this.dataTransferCallback = null;
            this.activeDevice = null;
            this.errorText = "";
        }

        // Getters and Setters
        public void setCurrentLogLevel(LogLevel logLevel) {
            this.currentLogLevel = logLevel;
        }

        public void setNanolibAccessor(NanoLibAccessor nanoLibAccessor) {
            this.nanolibAccessor = nanoLibAccessor;
        }

        public void setLoggingCallbackActive(boolean loggingCallbackActive) {
            this.loggingCallbackActive = loggingCallbackActive;
        }

        public void setLoggingCallback(LoggingCallbackExample loggingCallback) {
            this.loggingCallback = loggingCallback;
        }

        public void setScanBusCallback(ScanBusCallbackExample scanBusCallback) {
            this.scanBusCallback = scanBusCallback;
        }

        public void setDataTransferCallback(DataTransferCallbackExample dataTransferCallback) {
            this.dataTransferCallback = dataTransferCallback;
        }

        public void setWaitForUserConfirmation(boolean waitForUserConfirmation) {
            this.waitForUserConfirmation = waitForUserConfirmation;
        }

        public NanoLibAccessor getNanolibAccessor() {
            return this.nanolibAccessor;
        }
    }

    /**
     * @brief Return error number as string
     * 
     * @param number the error number
     * @return error number as String
     */
    public static String getErrorNumberString(long number) {
        int byteValue = (int) ((number & 0xff000000) >> 24);
        String resultString;

        switch (byteValue) {
            case 0: resultString = "    0: Watchdog Reset"; break;
            case 1: resultString = "    1: Input voltage (+Ub) too high"; break;
            case 2: resultString = "    2: Output current too high"; break;
            case 3: resultString = "    3: Input voltage (+Ub) too low"; break;
            case 4: resultString = "    4: Error at fieldbus"; break;
            case 6: resultString = "    6: CANopen only: NMT master takes too long to send Nodeguarding request"; break;
            case 7: resultString = "    7: Sensor 1 (see 3204h): Error through electrical fault or defective hardware"; break;
            case 8: resultString = "    8: Sensor 2 (see 3204h): Error through electrical fault or defective hardware"; break;
            case 9: resultString = "    9: Sensor 3 (see 3204h): Error through electrical fault or defective hardware"; break;
            case 10: resultString = "   10: Positive limit switch exceeded"; break;
            case 11: resultString = "   11: Negative limit switch exceeded"; break;
            case 12: resultString = "   12: Overtemperature error"; break;
            case 13: resultString = "   13: The values of object 6065h and 6066h were exceeded; a fault was triggered."; break;
            case 14: resultString = "   14: Nonvolatile memory full. Controller must be restarted for cleanup work."; break;
            case 15: resultString = "   15: Motor blocked"; break;
            case 16: resultString = "   16: Nonvolatile memory damaged; controller must be restarted for cleanup work."; break;
            case 17: resultString = "   17: CANopen only: Slave took too long to send PDO messages."; break;
            case 18: resultString = "   18: Sensor n (see 3204h), where n is greater than 3: Error through electrical fault or defective hardware"; break;
            case 19: resultString = "   19: CANopen only: PDO not processed due to a length error."; break;
            case 20: resultString = "   20: CANopen only: PDO length exceeded."; break;
            case 21: resultString = "   21: Restart the controller to avoid future errors when saving (nonvolatile memory full/corrupt)."; break;
            case 22: resultString = "   22: Rated current must be set (203Bh:01h/6075h)."; break;
            case 23: resultString = "   23: Encoder resolution, number of pole pairs and some other values are incorrect."; break;
            case 24: resultString = "   24: Motor current is too high, adjust the PI parameters."; break;
            case 25: resultString = "   25: Internal software error, generic."; break;
            case 26: resultString = "   26: Current too high at digital output."; break;
            case 27: resultString = "   27: CANopen only: Unexpected sync length."; break;
            case 30: resultString = "   30: Error in speed monitoring: slippage error too large."; break;
            case 32: resultString = "   32: Internal error: Correction factor for reference voltage missing in the OTP."; break;
            case 35: resultString = "   35: STO Fault: STO was requested but not via both STO inputs"; break;
            case 36: resultString = "   36: STO Changeover: STO was requested but not via both STO inputs."; break;
            case 37: resultString = "   37: STO Active: STO is active, it generates no torque or holding torque."; break;
            case 38: resultString = "   38: STO Self-Test: Error during self-test of the firmware. Contact Nanotec."; break;
            case 39: resultString = "   39: Error in the ballast configuration: Invalid/unrealistic parameters entered."; break;
            case 40: resultString = "   40: Ballast resistor thermally overloaded."; break;
            case 41: resultString = "   41: Only EtherCAT: Sync Manager Watchdog: The controller has not received any PDO data for an excessively long period of time."; break;
            case 46: resultString = "   46: Interlock error: Bit 3 in 60FDh is set to 0, the motor may not start."; break;
            case 48: resultString = "   48: Only CANopen: NMT status has been set to stopped."; break;
            default: resultString = "   " + byteValue + ": Unknown error number"; break;
        }

        return resultString;
    }

    /**
     * @brief Return error class as string
     * 
     * @param number the error number
     * @return error class as String
     */
    public static String getErrorClassString(long number) {
        int byteValue = (int) ((number & 0xff0000) >> 16);
        String resultString;

        switch (byteValue) {
            case 1: resultString = "    1: General error, always set in the event of an error."; break;
            case 2: resultString = "    2: Current."; break;
            case 4: resultString = "    4: Voltage."; break;
            case 8: resultString = "    8: Temperature."; break;
            case 16: resultString = "   16: Communication"; break;
            case 32: resultString = "   32: Relates to the device profile."; break;
            case 64: resultString = "   64: Reserved, always 0."; break;
            case 128: resultString = "  128: Manufacturer-specific."; break;
            default: resultString = "  " + byteValue + ": Unknown error class."; break;
        }

        return resultString;
    }

    /**
     * @brief Return error code as string
     * 
     * @param number the error number
     * @return error code as String
     */
    public static String getErrorCodeString(long number) {
        int wordValue = (int) (number & 0xffff);
        String resultString;

        switch (wordValue) {
            case 0x1000: resultString = "0x1000: General error."; break;
            case 0x2300: resultString = "0x2300: Current at the controller output too large."; break;
            case 0x3100: resultString = "0x3100: Overvoltage/undervoltage at controller input."; break;
            case 0x4200: resultString = "0x4200: Temperature error within the controller."; break;
            case 0x5440: resultString = "0x5440: Interlock error: Bit 3 in 60FDh is set to 0, the motor may not start."; break;
            case 0x6010: resultString = "0x6010: Software reset (watchdog)."; break;
            case 0x6100: resultString = "0x6100: Internal software error, generic."; break;
            case 0x6320: resultString = "0x6320: Rated current must be set (203Bh:01h/6075h)."; break;
            case 0x7110: resultString = "0x7110: Error in the ballast configuration: Invalid/unrealistic parameters entered."; break;
            case 0x7113: resultString = "0x7113: Warning: Ballast resistor thermally overloaded."; break;
            case 0x7121: resultString = "0x7121: Motor blocked."; break;
            case 0x7200: resultString = "0x7200: Internal error: Correction factor for reference voltage missing in the OTP."; break;
            case 0x7305: resultString = "0x7305: Sensor 1 (see 3204h) faulty."; break;
            case 0x7306: resultString = "0x7306: Sensor 2 (see 3204h) faulty."; break;
            case 0x7307: resultString = "0x7307: Sensor n (see 3204h), where n is greater than 2."; break;
            case 0x7600: resultString = "0x7600: Warning: Nonvolatile memory full or corrupt; restart the controller for cleanup work."; break;
            case 0x8100: resultString = "0x8100: Error during fieldbus monitoring."; break;
            case 0x8130: resultString = "0x8130: CANopen only: Life Guard error or Heartbeat error."; break;
            case 0x8200: resultString = "0x8200: CANopen only: Slave took too long to send PDO messages."; break;
            case 0x8210: resultString = "0x8210: CANopen only: PDO was not processed due to a length error."; break;
            case 0x8220: resultString = "0x8220: CANopen only: PDO length exceeded."; break;
            case 0x8240: resultString = "0x8240: CANopen only: unexpected sync length."; break;
            case 0x8400: resultString = "0x8400: Error in speed monitoring: slippage error too large."; break;
            case 0x8611: resultString = "0x8611: Position monitoring error: Following error too large."; break;
            case 0x8612: resultString = "0x8612: Position monitoring error: Limit switch exceeded."; break;
            default: resultString = wordValue + ": Unknown error code."; break;
        }

        return resultString;
    }

    /**
     * @brief Create bus hardware options object from given bus hardware id
     * 
     * @param busHwId The id of the bus hardware taken from GetHardware()
     * @return BusHardwareOptions
     */
	public static BusHardwareOptions createBusHardwareOptions(BusHardwareId busHwId) {
		// create new bus hardware options
		BusHardwareOptions busHwOptions = new BusHardwareOptions();

		// now add all options necessary for opening the bus hardware
		// in case of CAN bus it is the baud rate
		BusHwOptionsDefault busHwOptionsDefaults = new BusHwOptionsDefault();

		// now add all options necessary for opening the bus hardware
		if (busHwId.getProtocol().equals(Nanolib.getBUS_HARDWARE_ID_PROTOCOL_CANOPEN())) {
			// in case of CAN bus it is the baud rate
			busHwOptions.addOption(busHwOptionsDefaults.getCanBus().getBAUD_RATE_OPTIONS_NAME(),
					busHwOptionsDefaults.getCanBus().getBaudRate().getBAUD_RATE_1000K());

			if (busHwId.getBusHardware().equals(Nanolib.getBUS_HARDWARE_ID_IXXAT())) {
				// in case of HMS IXXAT we need also bus number
				busHwOptions.addOption(busHwOptionsDefaults.getCanBus().getIxxat().getADAPTER_BUS_NUMBER_OPTIONS_NAME(),
						busHwOptionsDefaults.getCanBus().getIxxat().getAdapterBusNumber().getBUS_NUMBER_0_DEFAULT());
			}

            if (busHwId.getBusHardware().equals(Nanolib.getBUS_HARDWARE_ID_PEAK())) {
				// in case of PEAK PCAN we need also bus number
				busHwOptions.addOption(busHwOptionsDefaults.getCanBus().getPeak().getADAPTER_BUS_NUMBER_OPTIONS_NAME(),
						busHwOptionsDefaults.getCanBus().getPeak().getAdapterBusNumber().getBUS_NUMBER_1_DEFAULT());
			}
		} else if (busHwId.getProtocol().equals(Nanolib.getBUS_HARDWARE_ID_PROTOCOL_MODBUS_RTU())) {
			// in case of Modbus RTU it is the serial baud rate
			busHwOptions.addOption(busHwOptionsDefaults.getSerial().getBAUD_RATE_OPTIONS_NAME(),
					busHwOptionsDefaults.getSerial().getBaudRate().getBAUD_RATE_19200());
			// and serial parity
			busHwOptions.addOption(busHwOptionsDefaults.getSerial().getPARITY_OPTIONS_NAME(),
					busHwOptionsDefaults.getSerial().getParity().getEVEN());
		}

		return busHwOptions;
	}

    /**
     * @brief Obtain a line of text from the defined reader.
     * 
     * @param def optional default text if no text entered
     * @return either valid input line or no value if problem obtaining input
     */
    public static Optional<String> getline(String def) {
        String line = "";
        try {
            line = reader.readLine().trim();
        } catch (Exception e) {
            // do nothing
        }
        
        return line.isEmpty() && !def.isEmpty() ? Optional.of(def) : Optional.of(line);
    }

    /**
     * @brief Obtain a line of text from console. 
     * 
     * @param prm optional prompt text to display first prm
     * @param def optional default text
     * @return entered text as type string. No error conditions. Only returns when valid data entered
     */
    public static String getline(String prm, String def) {
        Optional<String> input;

        do {
            System.out.print(prm);
            if (!def.isEmpty()) {
                System.out.print(" [" + def + "]");
            }
            System.out.print(": ");
            input = getline(def);

        } while (!input.isPresent());
        
        return input.get();
    }

    /**
     * @brief Obtains a number from the console. 
     * @details First displays prompt text. If defined, number must be within 
     *          the specified min..max range and range displayed as (...) after prm
     * 
     * @param prm optional prompt text to display first
     * @param nmin optional minimum valid value
     * @param nmax optional maximum valid value
     * @return returns when valid number entered
     */
    public static int getnum(String prm, int nmin, int nmax, boolean wholeLine) {
        Optional<Integer> result;

        do {
            System.out.print(prm);
            if (nmin != Integer.MIN_VALUE || nmax != Integer.MAX_VALUE) {
                System.out.print(" (" + nmin + " - " + nmax + ")");
            }
            System.out.print(": ");
            String inputResultString = "";
            
            try {
                inputResultString = reader.readLine().trim();

                if (inputResultString.isEmpty()) {
                    result = Optional.empty();
                } else {
                    result = Optional.of(Integer.valueOf(inputResultString));
                }
            } catch (Exception e) {
                return Integer.MAX_VALUE;
            }
            
            if (result.isPresent() && (result.get() < nmin || result.get() > nmax)) {
                result = Optional.of(Integer.MAX_VALUE);
            } 
        } while (!result.isPresent());

        return result.get();
    }

    /**
     * @brief Obtains a char from the defined reader
     * 
     * @param def default char to return if no character obtained
     * @return returns either valid character or no value if problem extracting data
     */
    public static Optional<Character> getchr(char def) {
        Optional<Character> result;
        String inputResultString = "";

        try {
            inputResultString = reader.readLine().trim();

            if (inputResultString.isEmpty()) {
                result = Optional.of(def);
            } else {
                result = Optional.of(inputResultString.charAt(0));
            }
        } catch (Exception e) {
            result = Optional.empty();
        }
        
        return result;
    }

    /**
     * @brief Obtains a char from the console. First displays prompt text
     * 
     * @param prm optional prompt text to display first
     * @param valid optional string containing valid values for the char. Displayed within (...)
     * @param def optional default char to use if none entered. Displayed within [...]
     * @return returns valid char. No error conditions. Only returns when valid char entered
     */
    public static char getchr(String prm, String valid, char def) {
        Optional<Character> result;

        do {
            System.out.print(prm);
            if (!valid.isEmpty()) {
                System.out.print(" (valid: " + valid + ")");
                if (def != 0) {
                    System.out.print(" [" + def + "]");
                }
            }
            System.out.print(": ");
            result = getchr(def);
        } while (!result.isPresent() || (!valid.isEmpty() && valid.indexOf(result.get()) == -1));

        return result.get();
    }

    /**
     * @brief Displays error message.
     * 
     * @param ctx the menu context
     * @param errorString general error or warning string, will be displayed in yellow
     * @param errorReasonString Abort error from application or Nanolib, will be displayed in red
     * @return returns the generated string
     */
    public static String handleErrorMessage(Context ctx, String errorString, String errorReasonString) {
        StringBuilder errorMessage = new StringBuilder();
        errorMessage.append(ctx.lightYellow.getEscapeCode()).append(errorString).append(ctx.lightRed.getEscapeCode()).append(errorReasonString).append(ctx.def.getEscapeCode());
        ctx.errorText = errorMessage.toString();

        if (ctx.waitForUserConfirmation) {
            System.out.println(errorMessage);
        }
        
        return errorMessage.toString();
    }

    /**
     * @brief Menu class
     */
    static class Menu {
        /**
         * @brief Functional inteface Variant
         */
        @FunctionalInterface
        interface Variant {
            void execute(Context ctx);
        }

        /**
         * @brief Inner class for menu items
         */
        static class MenuItem {
            String name;
            Variant func; // Pointer to a function or null for menu
            Menu subMenu;
            boolean isActive;

            /**
             * @brief Menu item construcotr
             * 
             * @param name name (title) of the menu item
             * @param func function to execute via functional interface
             */
            MenuItem(String name, Variant func) {
                this.name = name;
                this.func = func;
                this.subMenu = null;
                this.isActive = false;
            }

            /**
             * @brief Menu item construcotr
             * 
             * @param name name (title) of the menu item
             * @param func function to execute via functional interface
             * @param subMenu menu to activate if selected
             */
            MenuItem(String name, Variant func, Menu subMenu) {
                this.name = name;
                this.func = func;
                this.subMenu = subMenu;
                this.isActive = false;
            }

            /**
             * @brief Menu item construcotr
             * 
             * @param name name (title) of the menu item
             * @param func function to execute via functional interface
             * @param subMenu menu to activate if selected
             * @param isActive menu item active or inactive
             */
            MenuItem(String name, Variant func, Menu subMenu, boolean isActive) {
                this.name = name;
                this.func = func;
                this.subMenu = subMenu;
                this.isActive = isActive;
            }
        }

        private String title; // Menu title
        private List<MenuItem> menuItems; // List of menu items
        private Variant defaultFunc; // Default function for dynamic menu

        /**
         * @brief Default constructor
         */
        public Menu() {
            this.title = "";
            this.defaultFunc = null;
            this.menuItems = new ArrayList<>();
        }

        /**
         * @brief Constructor with parameters
         * 
         * @param title name (title) of the menu
         * @param menuItems list of menu items for this menu
         * @param defaultFunc default function to use for menu items in dynamic menu
         */
        public Menu(String title, List<MenuItem> menuItems, Variant defaultFunc) {
            this.title = title;
            this.menuItems = menuItems != null ? menuItems : new ArrayList<>();
            this.defaultFunc = defaultFunc;
        }

        /**
         * @brief Get the title of a menu
         * 
         * @return returns the menu title as String
         */
        public String getTitle() {
            return title;
        }

        /**
         * @brief Set a title of a menu
         * 
         * @param title the menu title
         */
        public void setTitle(String title) {
            this.title = title;
        }

        /**
         * @brief Get the configured default function for dynamic menu
         * 
         * @return returns the default function (functional interface)
         */
        public Variant getDefaultFunction() {
            return defaultFunc;
        }

        /**
         * @brief Display menu
         * 
         * @param ctx the menu context
         */
        public void menu(Context ctx) {
            menu(this, ctx);
        }

        /**
         * @brief Delete a menu item
         * 
         * @param index the index of the menu item in the menu item list
         * @return returns true if erased, false otherwise
         */
        public boolean eraseMenuItem(int index) {
            if (index < menuItems.size()) {
                menuItems.remove(index);
                return true;
            }
            return false;
        }

        /**
         * @brief Delete all configured menu items
         * 
         * @return returns true
         */
        public boolean eraseAllMenuItems() {
            menuItems.clear();
            return true;
        }

        /**
         * @brief Add a menu item
         * 
         * @param menuItem the menu item to add, no checks done
         * @return returns true
         */
        public boolean appendMenuItem(MenuItem menuItem) {
            menuItems.add(menuItem);
            return true;
        }

        /**
         * @brief Insert a menu item at index
         * 
         * @param index the index of the menu item in the menu item list
         * @param menuItem the menu item to insert
         * @return returns true if successfully inserted, false otherwise
         */
        public boolean insertMenuItem(int index, MenuItem menuItem) {
            if (index < menuItems.size()) {
                menuItems.add(index, menuItem);
                return true;
            }
            return false;
        }

        /**
         * @brief Get all found devices not yet connected
         * 
         * @param ctx the menu context
         * @return returns a list of DeviceId for all found devices not yet connected
         */
        public static List<DeviceId> getConnectableDeviceIds(Context ctx) {
            List<DeviceId> result = new ArrayList<>();
    
            for (DeviceId scannedDeviceId : ctx.scannedDeviceIds) {
                boolean alreadyConnected = false;
    
                for (DeviceHandle connectedDeviceHandle : ctx.connectedDeviceHandles) {
                    DeviceId connectedDeviceId = ctx.nanolibAccessor.getDeviceId(connectedDeviceHandle).getResult();
                    if (connectedDeviceId.equals(scannedDeviceId)) {
                        alreadyConnected = true;
                        break;
                    }
                }
    
                if (!alreadyConnected) {
                    result.add(scannedDeviceId);
                }
            }
    
            return result;
        }

        /**
         * @brief Get all found bus harware ids not yet opened
         * 
         * @param ctx the menu context
         * @return returns a list of BusHardwareId for all found bus hardware ids not yet opened 
         */
        public static List<BusHardwareId> getOpenableBusHwIds(Context ctx) {
            List<BusHardwareId> result = new ArrayList<>();
    
            for (BusHardwareId scannedBusHw : ctx.scannedBusHardwareIds) {
                boolean alreadyOpened = false;
    
                for (BusHardwareId openBusHwId : ctx.openBusHardwareIds) {
                    if (openBusHwId.equals(scannedBusHw)) {
                        alreadyOpened = true;
                        break;
                    }
                }
    
                if (!alreadyOpened) {
                    result.add(scannedBusHw);
                }
            }
    
            return result;
        }

        /**
         * @brief Sets the default menu items with given name and default function for dynamic menus
         * 
         * @param menu the menu to modify
         * @param ctx the menu context
         */
        public static void setMenuItems(Menu menu, Context ctx) {
        if (menu.getTitle().equals(NlcConstants.MAIN_MENU)) {
            for (MenuItem mi : menu.menuItems) {
                switch (mi.name) {
                    case NlcConstants.BUS_HARDWARE_MENU:
                        mi.isActive = true;
                        break;
                    case NlcConstants.DEVICE_MENU:
                        mi.isActive = !ctx.openBusHardwareIds.isEmpty();
                        break;
                    case NlcConstants.OD_INTERFACE_MENU:
                    case NlcConstants.SAMPLER_EXAMPLE_MENU:
                    case NlcConstants.MOTOR_EXAMPLE_MENU:
                    case NlcConstants.PROFINET_EXAMPLE_MI:
                        mi.isActive = (ctx.activeDevice != null) && (!ctx.activeDevice.equals(new DeviceHandle()));
                        break;
                    case NlcConstants.LOGGING_MENU:
                        mi.isActive = true;
                        break;
                }
            }
            } else if (menu.getTitle().equals(NlcConstants.BUS_HARDWARE_MENU)) {
                for (MenuItem mi : menu.menuItems) {
                    switch (mi.name) {
                        case NlcConstants.BUS_HARDWARE_SCAN_MI:
                            mi.isActive = true;
                            break;
                        case NlcConstants.BUS_HARDWARE_OPEN_MI:
                            mi.isActive = !ctx.openableBusHardwareIds.isEmpty();
                            break;
                        case NlcConstants.BUS_HARDWARE_CLOSE_MI:
                        case NlcConstants.BUS_HARDWARE_CLOSE_ALL_MI:
                            mi.isActive = !ctx.openBusHardwareIds.isEmpty();
                            break;
                    }
                }
            } else if (menu.getTitle().equals(NlcConstants.DEVICE_MENU)) {
                for (MenuItem mi : menu.menuItems) {
                    switch (mi.name) {
                        case NlcConstants.DEVICE_SCAN_MI:
                            mi.isActive = !ctx.openBusHardwareIds.isEmpty();
                            break;
                        case NlcConstants.DEVICE_CONNECT_MENU:
                            mi.isActive = !ctx.connectableDeviceIds.isEmpty() && !ctx.openBusHardwareIds.isEmpty();
                            break;
                        case NlcConstants.DEVICE_DISCONNECT_MENU:
                            mi.isActive = !ctx.connectedDeviceHandles.isEmpty();
                            break;
                        case NlcConstants.DEVICE_SELECT_ACTIVE_MENU:
                            mi.isActive = !ctx.connectedDeviceHandles.isEmpty() && !ctx.openBusHardwareIds.isEmpty();
                            break;
                        default:
                            mi.isActive = (ctx.activeDevice != null) && !ctx.activeDevice.equals(new DeviceHandle()) && (
                                    mi.name.equals(NlcConstants.DEVICE_INFORMATION_MENU) ||
                                    mi.name.equals(NlcConstants.DEVICE_REBOOT_MI) ||
                                    mi.name.equals(NlcConstants.DEVICE_UPDATE_FW_MI) ||
                                    mi.name.equals(NlcConstants.DEVICE_UPDATE_BL_MI) ||
                                    mi.name.equals(NlcConstants.DEVICE_UPLOAD_NANOJ_MI) ||
                                    mi.name.equals(NlcConstants.DEVICE_RUN_NANOJ_MI) ||
                                    mi.name.equals(NlcConstants.DEVICE_STOP_NANOJ_MI) ||
                                    mi.name.equals(NlcConstants.DEVICE_GET_ERROR_FIELD_MI) ||
                                    mi.name.equals(NlcConstants.DEVICE_RESTORE_ALL_DEFAULT_PARAMS_MI));
                            break;
                    }
                }
            } else if (menu.getTitle().equals(NlcConstants.DEVICE_INFORMATION_MENU) ||
                    menu.getTitle().equals(NlcConstants.OD_INTERFACE_MENU) ||
                    menu.getTitle().equals(NlcConstants.SAMPLER_EXAMPLE_MENU) ||
                    menu.getTitle().equals(NlcConstants.MOTOR_EXAMPLE_MENU)) {
                for (MenuItem mi : menu.menuItems) {
                    mi.isActive = (ctx.activeDevice != null) && !ctx.activeDevice.equals(new DeviceHandle());
                }
            } else if (menu.getTitle().equals(NlcConstants.LOG_LEVEL_MENU) ||
                    menu.getTitle().equals(NlcConstants.LOGGING_MENU) ||
                    menu.getTitle().equals(NlcConstants.LOG_CALLBACK_MENU)) {
                for (MenuItem mi : menu.menuItems) {
                    mi.isActive = true;
                }
            } else if (menu.getTitle().equals(NlcConstants.BUS_HARDWARE_OPEN_MI)) {
                menu.eraseAllMenuItems();
                List<BusHardwareId> openableBusHardwareIds = getOpenableBusHwIds(ctx);

                for (BusHardwareId openableBusHwId : openableBusHardwareIds) {
                    MenuItem mi = new MenuItem(openableBusHwId.getProtocol() + " (" + openableBusHwId.getName() + ")", menu.getDefaultFunction());
                    mi.isActive = true;
                    menu.appendMenuItem(mi);
                }
            } else if (menu.getTitle().equals(NlcConstants.BUS_HARDWARE_CLOSE_MI)) {
                menu.eraseAllMenuItems();
                List<BusHardwareId> openBusHwIds = ctx.openBusHardwareIds;

                for (BusHardwareId openBusHwId : openBusHwIds) {
                    MenuItem mi = new MenuItem(openBusHwId.getProtocol() + " (" + openBusHwId.getBusHardware() + ")", menu.getDefaultFunction());
                    mi.isActive = true;
                    menu.appendMenuItem(mi);
                }
            } else if (menu.getTitle().equals(NlcConstants.DEVICE_CONNECT_MENU)) {
                menu.eraseAllMenuItems();
                List<DeviceId> connectableDeviceIds = getConnectableDeviceIds(ctx);

                for (DeviceId connectableDeviceId : connectableDeviceIds) {
                    MenuItem mi = new MenuItem(connectableDeviceId.getDescription() +
                    " [id: " + connectableDeviceId.getDeviceId() +
                    ", protocol: " + connectableDeviceId.getBusHardwareId().getProtocol() +
                    ", hw: " + connectableDeviceId.getBusHardwareId().getName() + "]", menu.getDefaultFunction());
                    mi.isActive = true;
                    menu.appendMenuItem(mi);
                }
            } else if (menu.getTitle().equals(NlcConstants.DEVICE_DISCONNECT_MENU)) {
                List<DeviceId> openDeviceIds = new ArrayList<>();
                menu.eraseAllMenuItems();

                for (DeviceHandle openDeviceHandle : ctx.connectedDeviceHandles) {
                    ResultDeviceId openDeviceIdResult = ctx.nanolibAccessor.getDeviceId(openDeviceHandle);
                    if (!openDeviceIdResult.hasError()) {
                        openDeviceIds.add(openDeviceIdResult.getResult());
                    }
                }

                for (DeviceId deviceId : openDeviceIds) {
                    MenuItem mi = new MenuItem(deviceId.getDescription() +
                    " [id: " + deviceId.getDeviceId() +
                    ", protocol: " + deviceId.getBusHardwareId().getProtocol() +
                    ", hw: " + deviceId.getBusHardwareId().getName() + "]", menu.getDefaultFunction());
                    mi.isActive = true;
                    menu.appendMenuItem(mi);
                }
            } else if (menu.getTitle().equals(NlcConstants.DEVICE_SELECT_ACTIVE_MENU)) {
                menu.eraseAllMenuItems();
                for (DeviceHandle connectedDeviceHandle : ctx.connectedDeviceHandles) {
                    ResultDeviceId deviceIdResult = ctx.nanolibAccessor.getDeviceId(connectedDeviceHandle);
                    if (!deviceIdResult.hasError()) {
                        DeviceId deviceId = deviceIdResult.getResult();
                        MenuItem mi = new MenuItem(deviceId.getDescription() +
                        " [id: " + deviceId.getDeviceId() +
                        ", protocol: " + deviceId.getBusHardwareId().getProtocol() +
                        ", hw: " + deviceId.getBusHardwareId().getName() + "]", menu.getDefaultFunction());
                        mi.isActive = true;
                        menu.appendMenuItem(mi);
                    }
                }
            }
        }

        /**
         * @brief Build the active device string for printInfo
         * 
         * @param ctx the menu context
         * @return returns the generated string
         */
        public String getActiveDeviceString(Context ctx) {
            StringBuilder result = new StringBuilder();
            
            if (ctx.activeDevice == null) {
                result.append("Active device    : ").append(ctx.darkGray.getEscapeCode()).append("None").append(ctx.def.getEscapeCode()).append("\n");
                return result.toString();
            }

            DeviceHandle activeDeviceHandle = ctx.activeDevice;
            if (activeDeviceHandle != null) {
                DeviceId deviceId = ctx.nanolibAccessor.getDeviceId(activeDeviceHandle).getResult();
                result.append("Active device    : ").append(ctx.lightGreen.getEscapeCode())
                    .append(deviceId.getDescription()).append(" [id: ")
                    .append(deviceId.getDeviceId()).append(ctx.def.getEscapeCode()).append("]\n");
            }
            return result.toString();
        }

        /**
         * @brief Build the number of found bus hardware for printInfo
         * 
         * @param ctx the menu context
         * @return returns the generated string
         */
        public String getFoundBusHwString(Context ctx) {
            StringBuilder result = new StringBuilder();
            result.append("Bus HW found     : ").append(ctx.darkGray.getEscapeCode()).append("None (not scanned?)").append(ctx.def.getEscapeCode()).append("\n");

            if (ctx.scannedBusHardwareIds.isEmpty()) {
                return result.toString();
            }

            result.setLength(0); // Clear the StringBuilder
            result.append("Bus HW found     : ").append(ctx.lightGreen.getEscapeCode()).append(ctx.scannedBusHardwareIds.size()).append(ctx.def.getEscapeCode()).append("\n");
            return result.toString();
        }

        /**
         * @brief Build the opened bus hardware string for printInfo
         * 
         * @param ctx the menu context
         * @return returns the generated string
         */
        public String getOpenedBusHwIdString(Context ctx) {
            StringBuilder result = new StringBuilder();
            result.append("Open Bus HW      : ").append(ctx.darkGray.getEscapeCode()).append("None").append(ctx.def.getEscapeCode()).append("\n");

            if (ctx.openBusHardwareIds.isEmpty()) {
                return result.toString();
            }

            boolean firstItem = true;
            result.setLength(0); // Clear the StringBuilder
            result.append("Open Bus HW      : ");

            for (BusHardwareId openBusHardwareId : ctx.openBusHardwareIds) {
                if (firstItem) {
                    result.append(ctx.lightGreen.getEscapeCode())
                        .append(openBusHardwareId.getProtocol())
                        .append(" (").append(openBusHardwareId.getName()).append(")").append(ctx.def.getEscapeCode());
                    firstItem = false;
                } else {
                    result.append(", ").append(ctx.lightGreen.getEscapeCode())
                        .append(openBusHardwareId.getProtocol())
                        .append(" (").append(openBusHardwareId.getName()).append(")").append(ctx.def.getEscapeCode());
                }
            }

            result.append("\n");
            return result.toString();
        }

        /**
         * @brief Build the number of found devices printInfo
         * 
         * @param ctx the menu context
         * @return returns the generated string
         */
        public String getScannedDeviceIdsString(Context ctx) {
            StringBuilder result = new StringBuilder();
            result.append("Device(s) found  : ").append(ctx.darkGray.getEscapeCode()).append("None (not scanned?)").append(ctx.def.getEscapeCode()).append("\n");

            if (ctx.scannedDeviceIds.isEmpty()) {
                return result.toString();
            }
            result.setLength(0); // Clear the StringBuilder
            result.append("Device(s) found  : ").append(ctx.lightGreen.getEscapeCode()).append(ctx.scannedDeviceIds.size()).append(ctx.def.getEscapeCode()).append("\n");
            return result.toString();
        }

        /**
         * @brief Build the connected device(s) string for printInfo
         * 
         * @param ctx the menu context
         * @return returns the generated string
         */
        public String getConnectedDevicesString(Context ctx) {
            StringBuilder result = new StringBuilder();
            result.append("Connected devices: ").append(ctx.darkGray.getEscapeCode()).append("None").append(ctx.def.getEscapeCode()).append("\n");

            if (ctx.connectedDeviceHandles.isEmpty()) {
                return result.toString();
            }

            boolean firstItem = true;
            result.setLength(0); // Clear the StringBuilder
            result.append("Connected devices: ");

            for (DeviceHandle connectedDeviceHandle : ctx.connectedDeviceHandles) {
                ResultDeviceId resultDeviceId = ctx.nanolibAccessor.getDeviceId(connectedDeviceHandle);
                if (resultDeviceId.hasError()) {
                    // Don't display
                    continue;
                }
                DeviceId connectedDeviceId = resultDeviceId.getResult();
                if (firstItem) {
                    result.append(ctx.lightGreen.getEscapeCode())
                        .append(connectedDeviceId.getDescription())
                        .append(" [id: ").append(connectedDeviceId.getDeviceId())
                        .append(", protocol: ").append(connectedDeviceId.getBusHardwareId().getProtocol())
                        .append(", hw: ").append(connectedDeviceId.getBusHardwareId().getName()).append("]").append(ctx.def.getEscapeCode());
                    firstItem = false;
                } else {
                    result.append(", ").append(ctx.lightGreen.getEscapeCode())
                        .append(connectedDeviceId.getDescription())
                        .append(" [id: ").append(connectedDeviceId.getDeviceId())
                        .append(", protocol: ").append(connectedDeviceId.getBusHardwareId().getProtocol())
                        .append(", hw: ").append(connectedDeviceId.getBusHardwareId().getName()).append("]").append(ctx.def.getEscapeCode());
                }
            }

            result.append("\n");
            return result.toString();
        }

        /**
         * @brief Build the callback logging string for printInfo
         * 
         * @param ctx the menu context
         * @return returns the generated string
         */
        public String getCallbackLoggingString(Context ctx) {
            StringBuilder result = new StringBuilder();
            result.append("Callback Logging : Off\n");

            if (!ctx.loggingCallbackActive) {
                return result.toString();
            }

            result.setLength(0); // Clear the StringBuilder
            result.append("Callback Logging : ").append(ctx.lightGreen.getEscapeCode()).append("On").append(ctx.def.getEscapeCode())
                .append(" (").append(LogModuleConverter.toString(ctx.currentLogModule)).append(")\n");
            return result.toString();
        }

        /**
         * @brief Build the object dictionary string for printInfo
         * 
         * @param ctx the menu context
         * @return returns the generated string
         */
        public String getObjectDictionaryString(Context ctx) {
            StringBuilder result = new StringBuilder();
            result.append("Object dictionary: ").append(ctx.darkGray.getEscapeCode()).append("Fallback (not assigned)").append(ctx.def.getEscapeCode()).append("\n");

            if (ctx.activeDevice == null) {
                return result.toString();
            }

            ResultObjectDictionary resultObjectDictionary = ctx.nanolibAccessor.getAssignedObjectDictionary(ctx.activeDevice);
            if (resultObjectDictionary.hasError()) {
                return result.toString();
            }

            ObjectDictionary objectDictionary = resultObjectDictionary.getResult();

            if (objectDictionary.getXmlFileName().getResult().isEmpty()) {
                return result.toString();
            }

            result.setLength(0); // Clear the StringBuilder
            result.append("Object dictionary: ").append(ctx.lightGreen.getEscapeCode()).append("Assigned").append(ctx.def.getEscapeCode()).append("\n");
            
            return result.toString();
        }

        /**
         * @brief Prints basic information for the user
         * 
         * @param ctx the menu context
         * @return the complete string for output
         */
        public String printInfo(Context ctx) {
            // Clear screen, return value not needed
            clearScreen();
            
            StringBuilder oss = new StringBuilder();
            
            oss.append(getActiveDeviceString(ctx));
            oss.append(getFoundBusHwString(ctx));
            oss.append(getOpenedBusHwIdString(ctx));
            oss.append(getScannedDeviceIdsString(ctx));
            oss.append(getConnectedDevicesString(ctx));
            oss.append(getCallbackLoggingString(ctx));
            oss.append(getObjectDictionaryString(ctx));
            oss.append("Log level        : ").append(LogLevelConverter.toString(ctx.currentLogLevel)).append("\n");
            
            // Color handling done in handleErrorMessage
            oss.append(ctx.errorText).append("\n");
            
            // Clear text
            ctx.errorText = "";
            
            return oss.toString();
        }

        /**
         * @brief Display the menu, wait and get user input
         * 
         * @param currentMenu the menu to display
         * @param ctx the menu context
         * @return returns the user selected option
         */
        static int showMenu(Menu currentMenu, Context ctx) {
            // Dynamic part (for some menus)
            setMenuItems(currentMenu, ctx);
            
            // Static part
            StringBuilder oss = new StringBuilder();
            int numberOfMenuItems = currentMenu.menuItems.size();
            
            // Wait for user confirmation if required
            if (ctx.waitForUserConfirmation) {
                System.out.println("Press enter to continue!");
                try {
                    reader.readLine();
                } catch (IOException e) {
                    // do nothing
                }
            }
            
            ctx.waitForUserConfirmation = false;
            
            // Create user information part
            oss.append(currentMenu.printInfo(ctx));
            
            // Create menu header
            oss.append("---------------------------------------------------------------------------\n");
            oss.append(" ").append(currentMenu.getTitle()).append("\n");
            oss.append("---------------------------------------------------------------------------\n");

            // Create menu items (options)
            for (int i = 1; i <= numberOfMenuItems; i++) {
                if (currentMenu.menuItems.get(i - 1).isActive) {
                    oss.append(ctx.def.getEscapeCode()).append(((numberOfMenuItems > 9 && i < 10) ? " " : "")).append(i).append(") ")
                    .append(currentMenu.menuItems.get(i - 1).name).append('\n');
                } else {
                    oss.append(ctx.darkGray.getEscapeCode()).append(((numberOfMenuItems > 9 && i < 10) ? " " : "")).append(i).append(") ")
                    .append(currentMenu.menuItems.get(i - 1).name).append(ctx.def.getEscapeCode()).append('\n');
                }
            }
            
            // Create back (sub-menu) or exit option (main menu)
            if (currentMenu.getTitle().equals(NlcConstants.MAIN_MENU)) {
                oss.append("\n").append((numberOfMenuItems > 9 ? " " : "")).append("0) Exit program\n\nEnter menu option number");
            } else {
                oss.append("\n").append((numberOfMenuItems > 9 ? " " : "")).append("0) Back\n\nEnter menu option number");
            }

            // Display created output to screen and wait for user input
            return MenuUtils.getnum(oss.toString(), 0, numberOfMenuItems, true);
        }


        /**
         * @brief Process a menu.
         * 
         * @details Clear the screen, show the menu provided, wait for user selection and if valid
         *          call the linked function or open the next menu
         * @param menu the menu to handle
         * @param ctx the menu context
         */
        public void menu(Menu menu, Context ctx) {
            // Clear screen, result not needed
            clearScreen();
        
            ctx.waitForUserConfirmation = false;
            
            for (int opt = 0; (opt = showMenu(menu, ctx)) > 0;) {
                if (opt == Integer.MAX_VALUE || !menu.menuItems.get(opt - 1).isActive) {
                    ctx.errorText = ctx.lightYellow.getEscapeCode() + "Invalid option" + ctx.def.getEscapeCode();
                } else {
                    ctx.errorText = "";
        
                    // Store selected option in context
                    ctx.selectedOption = opt;
        
                    MenuItem mi = menu.menuItems.get(opt - 1);
                    if (mi.func != null) {
                        Variant variant = mi.func;
                        variant.execute(ctx);
                    } else {
                        menu(mi.subMenu, ctx);
                    }
                }
            }
        }
        
        /**
         * @brief Helper function to do a system call to clear the screen of console
         */
        public static void clearScreen() {
            // Clear screen implementation
            try {
                if (System.getProperty("os.name").toLowerCase().contains("win")) {
                    new ProcessBuilder("cmd", "/c", "cls").inheritIO().start().waitFor();
                } else {
                    new ProcessBuilder("clear").inheritIO().start().waitFor();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }        
    }
}



