/**
* Nanotec Nanolib example
* Copyright (C) Nanotec GmbH & Co. KG - All Rights Reserved
*
* This product includes software developed by the
* Nanotec GmbH & Co. KG (http://www.nanotec.com/).
*
* The Nanolib interface headers and the examples source code provided are 
* licensed under the Creative Commons Attribution 4.0 Internaltional License. 
* To view a copy of this license, 
* visit https://creativecommons.org/licenses/by/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* The parts of the library provided in binary format are licensed under 
* the Creative Commons Attribution-NoDerivatives 4.0 International License. 
* To view a copy of this license, 
* visit http://creativecommons.org/licenses/by-nd/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*
* @file   DeviceFunctionsExample.java
*
* @brief  Definition of device specific functions
*
* @date   29-10-2024
*
* @author Michael Milbradt
*
*/
package com.nanotec.example.NanolibExample;

import java.io.IOException;

import com.nanotec.example.NanolibExample.MenuUtils.Context;
import com.nanotec.example.NanolibExample.MenuUtils.Menu;
import com.nanotec.example.NanolibExample.MenuUtils.NlcConstants;
import com.nanotec.nanolib.BusHardwareId;
import com.nanotec.nanolib.ResultDeviceIds;
import com.nanotec.nanolib.DeviceId;
import com.nanotec.nanolib.ResultDeviceHandle;
import com.nanotec.nanolib.DeviceHandle;
import com.nanotec.nanolib.ResultVoid;
import com.nanotec.nanolib.ResultInt;
import com.nanotec.nanolib.ResultString;
import com.nanotec.nanolib.ResultArrayByte;
import com.nanotec.nanolib.ResultConnectionState;
import com.nanotec.nanolib.OdIndex;

/**
 * @brief Container class for device functions
 */
public class DeviceFunctionsExample {

    /**
    * @brief Scans for valid devices on all opened bus hardware
    *
    * @param  ctx menu context
    * @return void
    */
    public static void scanDevices(Context ctx) {
        ctx.waitForUserConfirmation = false;
        boolean found = false;
        ctx.scannedDeviceIds.clear();

        // no bus hardware
        if (ctx.openBusHardwareIds.isEmpty()) {
            MenuUtils.handleErrorMessage(ctx, "No bus hardware available. Please scan and select a bus hardware first.", "");
            return;
        }

        // scan for every opened bus hardware
        for (BusHardwareId openBusHardwareId : ctx.openBusHardwareIds) {
            System.out.println("Scan devices for " + openBusHardwareId.getProtocol() + " (" + openBusHardwareId.getName() + ")");
            ResultDeviceIds resultDeviceIds = ctx.nanolibAccessor.scanDevices(openBusHardwareId, ctx.scanBusCallback);
            if (resultDeviceIds.hasError()) {
                MenuUtils.handleErrorMessage(ctx, "Error during device scan: ", resultDeviceIds.getError());
                continue;
            }

            if (!resultDeviceIds.getResult().isEmpty()) {
                found = true;
                ctx.scannedDeviceIds.addAll(resultDeviceIds.getResult());
            }
        }

        if (!found) {
            MenuUtils.handleErrorMessage(ctx, "No devices found. Please check your cabling, driver(s) and/or device(s).", "");
            return;
        }

        // update ctx.connectableDeviceIds
        ctx.connectableDeviceIds = Menu.getConnectableDeviceIds(ctx);
    }

    /**
    * @brief Adds device and connects to the selected device (ctx.selectedOption) within Nanolib
    *
    * @param  ctx menu context
    * @return void
    */
    public static void connectDevice(Context ctx) {
        ctx.waitForUserConfirmation = false;

        if (ctx.connectableDeviceIds.isEmpty()) {
            MenuUtils.handleErrorMessage(ctx, "No device available. Please scan for devices first.", "");
            return;
        }

        // check if selected device is already connected
        int index = ctx.selectedOption;
        DeviceId selectedDeviceId = ctx.connectableDeviceIds.get(index - 1);

        ResultDeviceHandle deviceHandleResult = ctx.nanolibAccessor.addDevice(selectedDeviceId);
        if (deviceHandleResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during connectDevice (addDevice): ", deviceHandleResult.getError());
            return;
        }

        DeviceHandle deviceHandle = deviceHandleResult.getResult();

        ResultVoid resultVoid = ctx.nanolibAccessor.connectDevice(deviceHandle);
        if (resultVoid.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during connectDevice: ", resultVoid.getError());
            ctx.nanolibAccessor.removeDevice(deviceHandle);
            return;
        }

        // store handle
        ctx.connectedDeviceHandles.add(deviceHandle);

        // update availableDeviceIds
        ctx.connectableDeviceIds = Menu.getConnectableDeviceIds(ctx);

        // update ctx.activeDevice to new connection
        ctx.activeDevice = deviceHandle;
    }

    /**
    * @brief Disconnect device and removes to the selected device (ctx.selectedOption) within Nanolib
    *
    * @param  ctx menu context
    * @return void
    */
    public static void disconnectDevice(Context ctx) {
        ctx.waitForUserConfirmation = false;
        int index = ctx.selectedOption;

        if (ctx.connectedDeviceHandles.isEmpty()) {
            MenuUtils.handleErrorMessage(ctx, "No device connected.", "");
            return;
        }

        // get selected device handle
        DeviceHandle closeDeviceHandle = ctx.connectedDeviceHandles.get(index - 1);

        // disconnect device in nanolib
        ResultVoid resultVoid = ctx.nanolibAccessor.disconnectDevice(closeDeviceHandle);
        if (resultVoid.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during disconnectDevice: ", resultVoid.getError());
            return;
        }

        // remove device from nanolib
        resultVoid = ctx.nanolibAccessor.removeDevice(closeDeviceHandle);
        if (resultVoid.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during disconnectDevice (removeDevice): ", resultVoid.getError());
            return;
        }

        // update ctx.connectedDeviceHandles
        ctx.connectedDeviceHandles.remove(closeDeviceHandle);

        // update ctx.connectableDeviceIds
        ctx.connectableDeviceIds = Menu.getConnectableDeviceIds(ctx);

        // clear ctx.activeDevice
        ctx.activeDevice = null; // Assuming DeviceHandle can be null
    }

    /**
    * @brief Select the device to use for all device specific functions in Nanolib
    *
    * @param  ctx menu context
    * @return void
    */
    public static void selectActiveDevice(Context ctx) {
        ctx.waitForUserConfirmation = false;

        int index = ctx.selectedOption;

        if (ctx.connectedDeviceHandles.isEmpty()) {
            MenuUtils.handleErrorMessage(ctx, "No connected devices. Connect a device first.", "");
            return;
        }

        ctx.activeDevice = ctx.connectedDeviceHandles.get(index - 1);
    }

    /**
    * @brief Reboots the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void rebootDevice(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice == null) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultVoid rebootResult = ctx.nanolibAccessor.rebootDevice(ctx.activeDevice);
        if (rebootResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during rebootDevice: ", rebootResult.getError());
        }
    }

    /**
    * @brief Update the firmware of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void updateFirmware(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice == null) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        String inputPath;
        String deviceName = ctx.nanolibAccessor.getDeviceName(ctx.activeDevice).getResult();
        String firmwareBuildId = ctx.nanolibAccessor.getDeviceFirmwareBuildId(ctx.activeDevice).getResult();

        System.out.println("Current firmware Build Id: " + firmwareBuildId);
        System.out.println("Please enter the full path to the firmware file (e.g. " + deviceName + "-FIR-vXXXX-BXXXXXXX.fw): ");
        
        // Reading user input
        inputPath = readUserInput();
        
        if (inputPath == null) {
            MenuUtils.handleErrorMessage(ctx, "Error during updateFirmware:", "Invalid path");
            return; // Handle input error
        }

        System.out.println("Do not interrupt the data connection or switch off the power until the update process has been finished!");
        ResultVoid uploadResult = ctx.nanolibAccessor.uploadFirmwareFromFile(ctx.activeDevice, inputPath, ctx.dataTransferCallback);
        if (uploadResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during updateFirmware: ", uploadResult.getError());
        }
    }

    /**
    * @brief Update the bootloader of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void updateBootloader(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice == null) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        String inputPath;
        String bootloaderBuildId = ctx.nanolibAccessor.getDeviceBootloaderBuildId(ctx.activeDevice).getResult();
        String bootloaderVersion = String.valueOf(ctx.nanolibAccessor.getDeviceBootloaderVersion(ctx.activeDevice).getResult() >> 16);

        System.out.println("Current bootloader Build Id: " + bootloaderBuildId);
        System.out.println("Bootloader version: " + bootloaderVersion);
        System.out.println("Please enter the full path to the bootloader file: ");
        
        // Reading user input
        inputPath = readUserInput();
        
        if (inputPath == null) {
            MenuUtils.handleErrorMessage(ctx, "Error during updateBootloader:", "Invalid path");
            return; // Handle input error
        }

        System.out.println("Do not interrupt the data connection or switch off the power until the update process has been finished!");
        ResultVoid uploadResult = ctx.nanolibAccessor.uploadBootloaderFromFile(ctx.activeDevice, inputPath, ctx.dataTransferCallback);
        if (uploadResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during updateBootloader: ", uploadResult.getError());
        }
    }

    /**
    * @brief Upload a compiled NanoJ binary to the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void uploadNanoJ(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice == null) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        String inputPath;
        System.out.println("Please enter the full path to the NanoJ file (e.g. vmmcode.usr): ");
        
        // Reading user input
        inputPath = readUserInput();
        
        if (inputPath == null) {
            MenuUtils.handleErrorMessage(ctx, "Error during uploadNanoJ:", "Invalid path");
            return; // Handle input error
        }

        System.out.println("Do not interrupt the data connection or switch off the power until the update process has been finished!");
        ResultVoid uploadResult = ctx.nanolibAccessor.uploadNanoJFromFile(ctx.activeDevice, inputPath, ctx.dataTransferCallback);
        if (uploadResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during uploadNanoJ: ", uploadResult.getError());
            return;
        }
        System.out.println("Use runNanoJ menu option to re-start the uploaded NanoJ program.");
    }
 
    /**
    * @brief Executes the NanoJ program on the current active device if available
    *
    * @param  ctx menu context
    * @return void
    */
    public static void runNanoJ(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice == null) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        // check for errors
        ResultInt errorResult = ctx.nanolibAccessor.readNumber(ctx.activeDevice, NlcConstants.OD_NANOJ_ERROR);
        if (errorResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during runNanoJ: ", errorResult.getError());
            return;
        }

        if (errorResult.getResult() != 0) {
            MenuUtils.handleErrorMessage(ctx, "Failed to start NanoJ program - NanoJ error code is set: ", String.valueOf(errorResult.getResult()));
            return;
        }

        // write start to NanoJ control object (0x2300)
        ResultVoid writeNumberResult = ctx.nanolibAccessor.writeNumber(ctx.activeDevice, 0x1, NlcConstants.OD_NANOJ_CONTROL, 32);
        if (writeNumberResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during runNanoJ: ", writeNumberResult.getError());
            return;
        }

        // start might take some time (up to 200ms)
        try {
            Thread.sleep(250);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }

        // check if running and no error
        errorResult = ctx.nanolibAccessor.readNumber(ctx.activeDevice, NlcConstants.OD_NANOJ_ERROR);
        if (errorResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during runNanoJ: ", errorResult.getError());
            return;
        }

        if (errorResult.getResult() != 0) {
            MenuUtils.handleErrorMessage(ctx, "Error during runNanoJ - program exited with error: ", String.valueOf(errorResult.getResult()));
            return;
        }

        // check if program is still running, stopped or has error
        ResultInt readNumberResult = ctx.nanolibAccessor.readNumber(ctx.activeDevice, NlcConstants.OD_NANOJ_STATUS);
        if (readNumberResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during runNanoJ: ", readNumberResult.getError());
            return;
        }

        if (readNumberResult.getResult() == 0) {
            System.out.println("NanoJ program stopped ...");
        } else if (readNumberResult.getResult() == 1) {
            System.out.println("NanoJ program running ...");
        } else {
            System.out.println("NanoJ program exited with error.");
        }
    }

    /**
    * @brief Stops the NanoJ program on the current active device if available
    *
    * @param  ctx menu context
    * @return void
    */
    public static void stopNanoJ(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultVoid writeNumberResult = ctx.nanolibAccessor.writeNumber(ctx.activeDevice, 0x00, NlcConstants.OD_NANOJ_CONTROL, 32);
        if (writeNumberResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during stopNanoJ: ", writeNumberResult.getError());
            return;
        }

        // stop might take some time
        try {
            Thread.sleep(50);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }

        ResultInt readNumberResult = ctx.nanolibAccessor.readNumber(ctx.activeDevice, NlcConstants.OD_NANOJ_STATUS);
        if (readNumberResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during stopNanoJ: ", readNumberResult.getError());
            return;
        }

        switch ((int)readNumberResult.getResult()) {
            case 0:
                System.out.println("NanoJ program stopped ...");
                break;
            case 1:
                System.out.println("NanoJ program still running ...");
                break;
            default:
                System.out.println("NanoJ program exited with error: " + ctx.nanolibAccessor.readNumber(ctx.activeDevice, NlcConstants.OD_NANOJ_ERROR).getResult());
                break;
        }
    }

    /**
    * @brief Read and output the device vendor id of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void getDeviceVendorId(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultInt resultInt = ctx.nanolibAccessor.getDeviceVendorId(ctx.activeDevice);
        if (resultInt.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during getDeviceVendorId: ", resultInt.getError());
            return;
        }

        System.out.println("Device vendor id = '" + resultInt.getResult() + "'");
    }

    /**
    * @brief Read and output the product code of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void getDeviceProductCode(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultInt resultInt = ctx.nanolibAccessor.getDeviceProductCode(ctx.activeDevice);
        if (resultInt.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during getDeviceProductCode: ", resultInt.getError());
            return;
        }

        System.out.println("Device product code = '" + resultInt.getResult() + "'");
    }

    /**
    * @brief Read and output the device name of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void getDeviceName(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultString resultString = ctx.nanolibAccessor.getDeviceName(ctx.activeDevice);
        if (resultString.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during getDeviceName: ", resultString.getError());
            return;
        }

        System.out.println("Device name = '" + resultString.getResult() + "'");
    }

    /**
    * @brief Read and output the hardware version of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void getDeviceHardwareVersion(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultString resultString = ctx.nanolibAccessor.getDeviceHardwareVersion(ctx.activeDevice);
        if (resultString.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during getDeviceHardwareVersion: ", resultString.getError());
            return;
        }

        System.out.println("Device hardware version = '" + resultString.getResult() + "'");
    }

    /**
    * @brief Read and output the firmware build id of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void getDeviceFirmwareBuildId(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultString resultString = ctx.nanolibAccessor.getDeviceFirmwareBuildId(ctx.activeDevice);
        if (resultString.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during getDeviceFirmwareBuildId: ", resultString.getError());
            return;
        }

        System.out.println("Device firmware build id = '" + resultString.getResult() + "'");
    }

    /**
    * @brief Read and output the bootloader build id of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void getDeviceBootloaderBuildId(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultString resultString = ctx.nanolibAccessor.getDeviceBootloaderBuildId(ctx.activeDevice);
        if (resultString.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during getDeviceBootloaderBuildId: ", resultString.getError());
            return;
        }

        System.out.println("Device bootloader build id = '" + resultString.getResult() + "'");
    }

    /**
    * @brief Read and output the serial number of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void getDeviceSerialNumber(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultString resultString = ctx.nanolibAccessor.getDeviceSerialNumber(ctx.activeDevice);
        if (resultString.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during getDeviceSerialNumber: ", resultString.getError());
            return;
        }

        System.out.println("Device serial number = '" + resultString.getResult() + "'");
    }

    /**
    * @brief Read and output the device unique id of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void getDeviceUid(Context ctx) {
        ctx.setWaitForUserConfirmation(true);

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultArrayByte resultArray = ctx.getNanolibAccessor().getDeviceUid(ctx.activeDevice);

        if (resultArray.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during getDeviceUid: ", resultArray.getError());
            return;
        }

        // Convert byte array to hex string
        StringBuilder s = new StringBuilder(resultArray.getResult().size() * 2); // Two digits per byte
        final String hex = "0123456789ABCDEF";

        for (short c : resultArray.getResult()) {
            s.append(hex.charAt((c >> 4) & 0x0F)); // Upper nibble
            s.append(hex.charAt(c & 0x0F));        // Lower nibble
        }

        System.out.println("Device unique id = '" + s.toString() + "'");
    }

    /**
    * @brief Read and output the bootloader version of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void getDeviceBootloaderVersion(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultInt resultInt = ctx.nanolibAccessor.getDeviceBootloaderVersion(ctx.activeDevice);
        if (resultInt.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during getDeviceBootloaderVersion: ", resultInt.getError());
            return;
        }

        System.out.println("Device bootloader version = '" + (resultInt.getResult() >> 16) + "'");
    }

    /**
    * @brief Read and output the hardware group of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void getDeviceHardwareGroup(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultInt resultInt = ctx.nanolibAccessor.getDeviceHardwareGroup(ctx.activeDevice);
        if (resultInt.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during getDeviceHardwareGroup: ", resultInt.getError());
            return;
        }

        System.out.println("Device hardware group = '" + resultInt.getResult() + "'");
    }

    /**
    * @brief Read and output the connection state of the current active device
    *
    * @param  ctx menu context
    * @return void
    */
    public static void getConnectionState(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultConnectionState resultConState = ctx.nanolibAccessor.getConnectionState(ctx.activeDevice);
        if (resultConState.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during getConnectionState: ", resultConState.getError());
            return;
        }

        String connectionState = resultConState.getResult().toString();

        System.out.println("Device connection state = '" + connectionState + "'");
    }

    /**
    * @brief Read and output error-stack
    *
    * @param  ctx menu context
    * @return void
    */
    public static void getErrorFields(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        ResultInt errorNumberResult = ctx.nanolibAccessor.readNumber(ctx.activeDevice, NlcConstants.OD_ERROR_COUNT);
        if (errorNumberResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during getErrorField: ", errorNumberResult.getError());
            return;
        }

        if (errorNumberResult.getResult() == 0) {
            System.out.println("Currently there are no errors.");
            return;
        }

        // not more than 8 errors max
        short numberOfErrors = (short)errorNumberResult.getResult();
        System.out.println("Currently there are " + numberOfErrors + " errors.");

        for (short i = 1; i <= numberOfErrors; i++) {
            OdIndex currentErrorField = new OdIndex(NlcConstants.OD_ERROR_STACK_INDEX, i);
            errorNumberResult = ctx.nanolibAccessor.readNumber(ctx.activeDevice, currentErrorField);
            if (errorNumberResult.hasError()) {
                MenuUtils.handleErrorMessage(ctx, "Error during getErrorField: ", errorNumberResult.getError());
                return;
            }

            System.out.println("- Error Number [" + i + "] = " + MenuUtils.getErrorNumberString(errorNumberResult.getResult()));
            System.out.println("- Error Class  [" + i + "] = " + MenuUtils.getErrorClassString(errorNumberResult.getResult()));
            System.out.println("- Error Code   [" + i + "] = " + MenuUtils.getErrorCodeString(errorNumberResult.getResult()));
        }
    }

    /**
    * @brief Reset encoder resolution interfaces, reset drive mode selection and finally restore all default parameters
    *
    * @param  ctx menu context
    * @return void
    */
    public static void restoreDefaults(Context ctx) {
        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "", "No active device set. Select an active device first.");
            return;
        }

        // Read Additional position encoder resolution interface #1
        ResultInt posEncoderResResult = ctx.nanolibAccessor.readNumber(ctx.activeDevice, NlcConstants.OD_POS_ENCODER_INCREMENTS_INTERFACE_1);
        if (!posEncoderResResult.hasError()) {
            System.out.println("Position encoder resolution - encoder increments feedback interface #1 = " + posEncoderResResult.getResult());
        }

        // Read Additional position encoder resolution interface #2
        posEncoderResResult = ctx.nanolibAccessor.readNumber(ctx.activeDevice, NlcConstants.OD_POS_ENCODER_INCREMENTS_INTERFACE_2);
        if (!posEncoderResResult.hasError()) {
            System.out.println("Position encoder resolution - encoder increments feedback interface #2 = " + posEncoderResResult.getResult());
        }

        // Read Additional position encoder resolution interface #3
        posEncoderResResult = ctx.nanolibAccessor.readNumber(ctx.activeDevice, NlcConstants.OD_POS_ENCODER_INCREMENTS_INTERFACE_3);
        if (!posEncoderResResult.hasError()) {
            System.out.println("Position encoder resolution - encoder increments feedback interface #3 = " + posEncoderResResult.getResult());
        }

        // Set interface values to zero
        ctx.nanolibAccessor.writeNumber(ctx.activeDevice, 0, NlcConstants.OD_POS_ENCODER_INCREMENTS_INTERFACE_1, 32);
        ctx.nanolibAccessor.writeNumber(ctx.activeDevice, 0, NlcConstants.OD_POS_ENCODER_INCREMENTS_INTERFACE_2, 32);
        ctx.nanolibAccessor.writeNumber(ctx.activeDevice, 0, NlcConstants.OD_POS_ENCODER_INCREMENTS_INTERFACE_3, 32);

        ResultInt subModeSelectResult = ctx.nanolibAccessor.readNumber(ctx.activeDevice, NlcConstants.OD_MOTOR_DRIVE_SUBMODE_SELECT);
        System.out.println("Motor drive submode select = " + subModeSelectResult.getResult());

        // Set motor drive submode select to zero
        ctx.nanolibAccessor.writeNumber(ctx.activeDevice, 0, NlcConstants.OD_MOTOR_DRIVE_SUBMODE_SELECT, 32);

        // Save all parameters to non-volatile memory
        ResultVoid writeResult = ctx.nanolibAccessor.writeNumber(ctx.activeDevice, 1702257011, NlcConstants.OD_STORE_ALL_PARAMS, 32);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during restoreDefaults: ", writeResult.getError());
            return;
        }

        // Wait until write has completed
        while (true) {
            ResultInt storeResult = ctx.nanolibAccessor.readNumber(ctx.activeDevice, NlcConstants.OD_STORE_ALL_PARAMS);
            if (storeResult.getResult() == 1) {
                break;
            }
        }

        // Reboot current active device
        System.out.println("Rebooting ...");
        ResultVoid rebootResult = ctx.nanolibAccessor.rebootDevice(ctx.activeDevice);
        if (rebootResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during restoreDefaults: ", rebootResult.getError());
        }

        // Restore all default parameters
        System.out.println("Restoring all default parameters ...");
        writeResult = ctx.nanolibAccessor.writeNumber(ctx.activeDevice, 1684107116, NlcConstants.OD_RESTORE_ALL_DEF_PARAMS, 32);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during restoreDefaults: ", writeResult.getError());
            return;
        }

        // Restore tuning default parameters
        System.out.println("Restoring tuning default parameters ...");
        writeResult = ctx.nanolibAccessor.writeNumber(ctx.activeDevice, 1684107116, NlcConstants.OD_RESTORE_TUNING_DEF_PARAMS, 32);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during restoreDefaults: ", writeResult.getError());
            return;
        }

        // Reboot current active device
        System.out.println("Rebooting ...");
        rebootResult = ctx.nanolibAccessor.rebootDevice(ctx.activeDevice);
        if (rebootResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during restoreDefaults: ", rebootResult.getError());
        }

        System.out.println("All done. Check for errors.");
    }

    /**
    * @brief Helper method to read user input
    *
    * @param  ctx menu context
    * @return void
    */
    private static String readUserInput() {
        String input = "";
        try {
            input = MenuUtils.reader.readLine();
        } catch (IOException e) {
            return null;
        }
        
        return input.isEmpty() ? null : input;
    }
}

