/**
* Nanotec Nanolib example
* Copyright (C) Nanotec GmbH & Co. KG - All Rights Reserved
*
* This product includes software developed by the
* Nanotec GmbH & Co. KG (http://www.nanotec.com/).
*
* The Nanolib interface headers and the examples source code provided are 
* licensed under the Creative Commons Attribution 4.0 Internaltional License. 
* To view a copy of this license, 
* visit https://creativecommons.org/licenses/by/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* The parts of the library provided in binary format are licensed under 
* the Creative Commons Attribution-NoDerivatives 4.0 International License. 
* To view a copy of this license, 
* visit http://creativecommons.org/licenses/by-nd/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*
* @file   LoggingCallbackExample.java
*
* @brief  Definition of logging callback class
*
* @date   29-10-2024
*
* @author Michael Milbradt
*
*/
package com.nanotec.example.NanolibExample;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import com.nanotec.nanolib.NanoLibAccessor;
import com.nanotec.nanolib.Nanolib;
import com.nanotec.nanolib.NlcLoggingCallback;
import com.nanotec.nanolib.LogLevelConverter;
import com.nanotec.nanolib.LogLevel;

/**
* @brief Implementation class of NlcLoggingCallback, handling the logging callback
*/
public class LoggingCallbackExample extends NlcLoggingCallback {

    /**
    * @brief Gets called during data transfer.
    *
    * IMPORTANT: the member function pointer of NlcLoggingCallback::callback gets
    * bound to a std::function pointer in between (because of SWIG not supporting std::function)
    * and has to be (manually) unset/deleted in the destructor / during garbage collection!
    *
    * @return void
    */
    
    protected void finalize() {
        NanoLibAccessor nanolibAccessor = Nanolib.getNanoLibAccessor();
        nanolibAccessor.unsetLoggingCallback();
    }

    /**
    * @brief Gets called during log calls, handle your stuff here (e.g. write to file or console)
    *
    * @param  payload_str the complete log payload as string
    * @param  formatted_str the formatted log message as string
    * @param  logger_name name of module that outputs the log message
    * @param  log_level the log level of the log message
    * @param  time_since_epoch time since epoch in milliseconds
    * @param  thread_id thread id 
    * @return void
    */
    @Override
    public void callback(String payload_str, String formatted_str, String logger_name, long log_level, java.math.BigInteger time_since_epoch, long thread_id) {
        int pos;
        // Find the position of the line separator
        pos = formatted_str.indexOf(System.lineSeparator());
        if (pos != -1) {
            formatted_str = formatted_str.substring(0, pos);
        }

        // Do your callback stuff here...
        System.out.println("----------------------------------------------------------------------------------");
        System.out.println("| Payload = '" + payload_str + "'");
        System.out.println("| Formatted string = '" + formatted_str + "'");
        System.out.println("| Logger name = '" + logger_name + "'");
        System.out.println("| nlc_log_level = '" + LogLevelConverter.toString(LogLevel.swigToEnum((int)log_level)) + "'");
        System.out.println("| Local Time = '" + timeSinceEpochToLocaltimeString(time_since_epoch.longValue()) + "'");
        System.out.println("| Thread id = '" + thread_id + "'");
        System.out.println("----------------------------------------------------------------------------------");
    }

    /**
     * @brief Used to convert time since epoch value to a local time string
     *
     * @param timeSinceEpochInMs - time in milliseconds as String
     */
    public String timeSinceEpochToLocaltimeString(long timeSinceEpochInMs) {
        Instant instant = Instant.ofEpochMilli(timeSinceEpochInMs);
        LocalDateTime dateTime = LocalDateTime.ofInstant(instant, ZoneId.systemDefault());

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("dd-MM-yyyy HH:mm:ss");
        String formattedDate = dateTime.format(formatter);
        long fractionalSeconds = timeSinceEpochInMs % 1000;

        return formattedDate + ":" + fractionalSeconds;
    }
}
