/**
* Nanotec Nanolib example
* Copyright (C) Nanotec GmbH & Co. KG - All Rights Reserved
*
* This product includes software developed by the
* Nanotec GmbH & Co. KG (http://www.nanotec.com/).
*
* The Nanolib interface headers and the examples source code provided are 
* licensed under the Creative Commons Attribution 4.0 Internaltional License. 
* To view a copy of this license, 
* visit https://creativecommons.org/licenses/by/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* The parts of the library provided in binary format are licensed under 
* the Creative Commons Attribution-NoDerivatives 4.0 International License. 
* To view a copy of this license, 
* visit http://creativecommons.org/licenses/by-nd/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*
* @file   ProfinetFunctionsExample.java
*
* @brief  Definition of Profinet specific functions
*
* @date   29-10-2024
*
* @author Michael Milbradt
*
*/
package com.nanotec.example.NanolibExample;

import com.nanotec.example.NanolibExample.MenuUtils.Context;
import com.nanotec.nanolib.BusHardwareId;
import com.nanotec.nanolib.NlcErrorCode;
import com.nanotec.nanolib.ProfinetDCP;
import com.nanotec.nanolib.ProfinetDevice;
import com.nanotec.nanolib.ResultProfinetDevices;
import com.nanotec.nanolib.ResultVoid;
import com.nanotec.nanolib.ProfinetDeviceVector;

/**
 * @brief Container class for profinet functions
 */
public class ProfinetFunctionsExample {

    /**
     * @brief  Function to demonstrate how to connect and blink profinet device(s)
     * @param ctx - menu context
     */
    public static void profinetDCPExample(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.openBusHardwareIds.isEmpty()) {
            MenuUtils.handleErrorMessage(ctx, "No hardware bus available. Open a proper hardware bus first.", "");
            return;
        }

        boolean foundProfinetDevice = false;

        // Check service availability - Npcap/WinPcap driver required
        ProfinetDCP profinetDCP = ctx.nanolibAccessor.getProfinetDCP();

        // Search for profinet on every open bus hardware
        for (BusHardwareId openBusHwId : ctx.openBusHardwareIds) {
            ResultVoid serviceResult = profinetDCP.isServiceAvailable(openBusHwId);
            if (serviceResult.hasError()) {
                // Ignore
                continue;
            }

            // Service available - scan for profinet devices
            System.out.println("Scanning " + openBusHwId.getName() + " for Profinet devices...");
            ResultProfinetDevices resultProfinetDevices = profinetDCP.scanProfinetDevices(openBusHwId);

            if (resultProfinetDevices.hasError() 
                && resultProfinetDevices.getErrorCode() != NlcErrorCode.TimeoutError) {
                System.out.println("Error during profinetDCPExample: " + resultProfinetDevices.getError());
                continue;
            }

            ProfinetDeviceVector profinetDevices = resultProfinetDevices.getResult();
            int numberOfProfinetDevices = profinetDevices.size();

            if (numberOfProfinetDevices < 1) {
                continue;
            }

            foundProfinetDevice = true;
            System.out.println(numberOfProfinetDevices + " Profinet device(s) found:");
            for (ProfinetDevice profinetDevice : profinetDevices) {
                System.out.printf("IP: %d.%d.%d.%d\tName: %s%n",
                        (profinetDevice.getIpAddress() >> 24) & 0xFF,
                        (profinetDevice.getIpAddress() >> 16) & 0xFF,
                        (profinetDevice.getIpAddress() >> 8) & 0xFF,
                        profinetDevice.getIpAddress() & 0xFF,
                        profinetDevice.getDeviceName());

                // Checking the IP address against the context of the current network configuration
                ResultVoid resultValid = profinetDCP.validateProfinetDeviceIp(openBusHwId, profinetDevice);
                System.out.println("\tDevice IP is " + (resultValid.hasError() ? " not " : "") + "valid in the current network.");

                // Trying to blink the device
                ResultVoid resultBlink = profinetDCP.blinkProfinetDevice(openBusHwId, profinetDevice);
                System.out.print("\tBlink the device ");
                if (resultBlink.hasError()) {
                    System.out.println("failed with error: " + resultBlink.getError());
                } else {
                    System.out.println("succeeded.");
                }
            }
        }

        if (!foundProfinetDevice) {
            MenuUtils.handleErrorMessage(ctx, "No Profinet devices found.", "");
        }
    }
}
