///
/// Nanotec Nanolib example
/// Copyright (C) Nanotec GmbH & Co. KG - All Rights Reserved
///
/// This product includes software developed by the
/// Nanotec GmbH & Co. KG (http://www.nanotec.com/).
///
/// The Nanolib interface headers and the examples source code provided are 
/// licensed under the Creative Commons Attribution 4.0 Internaltional License. 
/// To view a copy of this license, 
/// visit https://creativecommons.org/licenses/by/4.0/ or send a letter to 
/// Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
///
/// The parts of the library provided in binary format are licensed under 
/// the Creative Commons Attribution-NoDerivatives 4.0 International License. 
/// To view a copy of this license, 
/// visit http://creativecommons.org/licenses/by-nd/4.0/ or send a letter to 
/// Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
///
/// This program is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
///
/// <file_name>SamplerFunctionsExample.cs</file_name>
///
/// <summary>Definition of sampler specific functions</summary>
///
/// <date>29-10-2024</date>
///
/// <author>Michael Milbradt<author>
///

using Nlc;
using MenuUtils;
using SamplerExample;

namespace SamplerFunctionsExample
{
    public static class SamplerFunctions
    {
        /// <summary>
        /// Function to execute the sampler example in normal mode without notification callback.
        /// </summary>
        /// <param name="ctx">Menu context.</param>
        public static void ExecuteSamplerWithoutNotificationNormalMode(Context ctx)
        {
            ctx.WaitForUserConfirmation = true;

            if (ctx.ActiveDevice.Equals(new DeviceHandle()))
            {
                StringUtils.HandleErrorMessage(ctx, "No active device set. Select an active device first.");
                return;
            }

            Console.WriteLine("In normal mode the number of samples can be configured.");
            Console.WriteLine("In this example the sampler will run for 5 samples.");

            SamplerExampleClass samplerExample = new SamplerExampleClass(ctx, ctx.ActiveDevice);
            samplerExample.ProcessSamplerWithoutNotificationNormal();

            Console.WriteLine("Finished");
        }

        /// <summary>
        /// Function to execute the sampler example in repetitive mode without notification callback.
        /// </summary>
        /// <param name="ctx">Menu context.</param>
        public static void ExecuteSamplerWithoutNotificationRepetitiveMode(Context ctx)
        {
            ctx.WaitForUserConfirmation = true;

            if (ctx.ActiveDevice.Equals(new DeviceHandle()))
            {
                StringUtils.HandleErrorMessage(ctx, "No active device set. Select an active device first.");
                return;
            }

            Console.WriteLine("In repetitive mode the sampler runs until stopped.");
            Console.WriteLine("In this example the sampler will run for 4 iterations and then stop.");

            SamplerExampleClass samplerExample = new SamplerExampleClass(ctx, ctx.ActiveDevice);
            samplerExample.ProcessSamplerWithoutNotificationRepetitive();

            Console.WriteLine("Finished");
        }

        /// <summary>
        /// Function to execute the sampler example in continuous mode without notification callback.
        /// </summary>
        /// <param name="ctx">Menu context.</param>
        public static void ExecuteSamplerWithoutNotificationContinuousMode(Context ctx)
        {
            ctx.WaitForUserConfirmation = true;

            if (ctx.ActiveDevice.Equals(new DeviceHandle()))
            {
                StringUtils.HandleErrorMessage(ctx, "No active device set. Select an active device first.");
                return;
            }

            Console.WriteLine("In continuous mode the sampler runs until stopped.");
            Console.WriteLine("In this example the sampler will run for 10 samples and then stop.");

            SamplerExampleClass samplerExample = new SamplerExampleClass(ctx, ctx.ActiveDevice);
            samplerExample.ProcessSamplerWithoutNotificationContinuous();

            Console.WriteLine("Finished");
        }

        /// <summary>
        /// Function to execute the sampler example in normal mode with notification callback.
        /// </summary>
        /// <param name="ctx">Menu context.</param>
        public static void ExecuteSamplerWithNotificationNormalMode(Context ctx)
        {
            ctx.WaitForUserConfirmation = true;

            if (ctx.ActiveDevice.Equals(new DeviceHandle()))
            {
                StringUtils.HandleErrorMessage(ctx, "No active device set. Select an active device first.");
                return;
            }

            Console.WriteLine("In normal mode the number of samples can be configured.");
            Console.WriteLine("In this example the sampler will run for 5 samples.");

            SamplerExampleClass samplerExample = new SamplerExampleClass(ctx, ctx.ActiveDevice);
            samplerExample.ProcessSamplerWithNotificationNormal();

            Console.WriteLine("Finished");
        }

        /// <summary>
        /// Function to execute the sampler example in repetitive mode with notification callback.
        /// </summary>
        /// <param name="ctx">Menu context.</param>
        public static void ExecuteSamplerWithNotificationRepetitiveMode(Context ctx)
        {
            ctx.WaitForUserConfirmation = true;

            if (ctx.ActiveDevice.Equals(new DeviceHandle()))
            {
                StringUtils.HandleErrorMessage(ctx, "No active device set. Select an active device first.");
                return;
            }

            Console.WriteLine("In repetitive mode the sampler runs until stopped.");
            Console.WriteLine("In this example the sampler will run for 4 iterations and then stop.");

            SamplerExampleClass samplerExample = new SamplerExampleClass(ctx, ctx.ActiveDevice);
            samplerExample.ProcessSamplerWithNotificationRepetitive();

            Console.WriteLine("Finished");
        }

        /// <summary>
        /// Function to execute the sampler example in continuous mode with notification callback.
        /// </summary>
        /// <param name="ctx">Menu context.</param>
        public static void ExecuteSamplerWithNotificationContinuousMode(Context ctx)
        {
            ctx.WaitForUserConfirmation = true;

            if (ctx.ActiveDevice.Equals(new DeviceHandle()))
            {
                StringUtils.HandleErrorMessage(ctx, "No active device set. Select an active device first.");
                return;
            }

            Console.WriteLine("In continuous mode the sampler runs until stopped.");
            Console.WriteLine("In this example the sampler will run for 10 samples and then stop.");

            SamplerExampleClass samplerExample = new SamplerExampleClass(ctx, ctx.ActiveDevice);
            samplerExample.ProcessSamplerWithNotificationContinuous();

            Console.WriteLine("Finished");
        }
    }
}


